<?php
session_start();

if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header('Location: index.php');
    exit;
}

// Conectar a la base de datos
class SQLiteWrapperCodes {
    private $db;
    public function __construct($path) {
        try {
            $this->db = new SQLite3($path);
        } catch (Exception $e) {
            die("Error de conexión a la base de datos: " . $e->getMessage());
        }
    }
    public function select($table, $columns, $where = '', $order = '', $params = []) {
        $query = "SELECT $columns FROM $table";
        if ($where) $query .= " WHERE $where";
        if ($order) $query .= " ORDER BY $order";
        $stmt = $this->db->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $result = $stmt->execute();
        $rows = [];
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $rows[] = $row;
        }
        return $rows;
    }
    public function delete($table, $where, $params) {
        $query = "DELETE FROM $table WHERE $where";
        $stmt = $this->db->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        return $stmt->execute();
    }
    public function querySingle($query) {
        return $this->db->querySingle($query);
    }
    public function close() {
        $this->db->close();
    }
}

$api_db = new SQLiteWrapperCodes(__DIR__ . '/api/codes.db');

// Funciones para CSRF
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Variables para modales
$mensaje_modal = '';
$tipo_modal = 'success'; // success, danger, warning

// Eliminar un código individual
if (isset($_GET['eliminar'])) {
    $id = filter_input(INPUT_GET, 'eliminar', FILTER_VALIDATE_INT);
    if ($id) {
        try {
            $api_db->delete('codes', 'id = :id', [':id' => $id]);
            $mensaje_modal = '✅ Código eliminado exitosamente.';
            $tipo_modal = 'success';
        } catch (Exception $e) {
            $mensaje_modal = '❌ Error al eliminar el código: ' . htmlspecialchars($e->getMessage());
            $tipo_modal = 'danger';
        }
    } else {
        $mensaje_modal = '❌ ID inválido.';
        $tipo_modal = 'danger';
    }
}

// Eliminar todos los usados
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'eliminar_usados') {
    if (!verify_csrf_token($_POST['csrf_token'])) {
        $mensaje_modal = '❌ Error de validación CSRF.';
        $tipo_modal = 'danger';
    } else {
        try {
            $count = $api_db->querySingle("SELECT COUNT(*) FROM codes WHERE estado = 1");
            if ($count > 0) {
                $api_db->delete('codes', 'estado = :estado', [':estado' => 1]);
                $mensaje_modal = '✅ Todos los códigos usados fueron eliminados.';
                $tipo_modal = 'success';
            } else {
                $mensaje_modal = '⚠️ No hay códigos usados para eliminar.';
                $tipo_modal = 'warning';
            }
        } catch (Exception $e) {
            $mensaje_modal = '❌ Error al eliminar códigos usados: ' . htmlspecialchars($e->getMessage());
            $tipo_modal = 'danger';
        }
    }
}

// Obtener todos los códigos
try {
    $codes = $api_db->select('codes', '*', '', 'created_at DESC');
} catch (Exception $e) {
    $mensaje_modal = '❌ Error al obtener los códigos: ' . htmlspecialchars($e->getMessage());
    $tipo_modal = 'danger';
    $codes = [];
}

include('includes/header.php');
?>

<style>
    .table-responsive { overflow-x: auto; }
    .table th, .table td { white-space: nowrap; }
    @media (max-width: 576px) {
        .table th, .table td { font-size: 0.8rem; padding: 6px; }
        .btn-sm { font-size: 0.75rem; padding: 4px 8px; }
    }
    .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.5); }
    .modal-content { background-color: secundary; margin: 15% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 500px; border-radius: 5px; }
    .modal-header { padding: 10px; border-bottom: 1px solid #dee2e6; }
    .modal-header.bg-success { background-color: #28a745; color: white; }
    .modal-header.bg-danger { background-color: #dc3545; color: white; }
    .modal-header.bg-warning { background-color: #ffc107; color: black; }
    .modal-body { padding: 15px; }
    .modal-footer { padding: 10px; border-top: 1px solid #dee2e6; text-align: right; }
    .btn { padding: 6px 12px; border: none; border-radius: 4px; cursor: pointer; }
    .btn-secondary { background-color: #6c757d; color: white; }
    .btn-danger { background-color: #dc3545; color: white; }
    .btn:hover { opacity: 0.9; }
</style>

<div class="col-md-10 mx-auto mt-4">
    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3>Gestionar Códigos</h3>
        </div>
        <div class="card-body">
            <!-- Formulario para eliminar usados -->
            <form method="POST" id="eliminarUsadosForm" class="mb-3">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars(generate_csrf_token()); ?>">
                <input type="hidden" name="action" value="eliminar_usados">
                <button type="submit" class="btn btn-danger">Eliminar Usados Masivamente</button>
            </form>

            <!-- Tabla de códigos -->
            <div class="table-responsive">
                <table class="table table-bordered table-striped">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>MAC Address</th>
                            <th>Código</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($codes as $row) { ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['id']); ?></td>
                                <td><?php echo htmlspecialchars($row['mac_address']); ?></td>
                                <td><?php echo htmlspecialchars($row['codigo']); ?></td>
                                <td>
                                    <?php if ($row['estado'] == 1) { ?>
                                        <span class="badge bg-success">Usado</span>
                                    <?php } else { ?>
                                        <span class="badge bg-warning text-dark">No Usado</span>
                                    <?php } ?>
                                </td>
                                <td>
                                    <button class="btn btn-sm btn-info" onclick="copiarCodigo('<?php echo htmlspecialchars($row['codigo']); ?>', this)">Copiar</button>
                                    <button class="btn btn-sm btn-danger" onclick="abrirModalEliminar(<?php echo $row['id']; ?>)">Eliminar</button>
                                </td>
                            </tr>
                        <?php } ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Modal de mensaje -->
<div id="mensajeModal" class="modal" style="<?php echo $mensaje_modal ? 'display: block;' : 'display: none;'; ?>">
    <div class="modal-content">
        <div class="modal-header bg-<?php echo $tipo_modal; ?>">
            <h5>Mensaje</h5>
        </div>
        <div class="modal-body">
            <?php echo htmlspecialchars($mensaje_modal); ?>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="cerrarModal('mensajeModal')">Cerrar</button>
        </div>
    </div>
</div>

<!-- Modal de confirmación eliminar -->
<div id="confirmEliminarModal" class="modal">
    <div class="modal-content">
        <div class="modal-header bg-danger">
            <h5>Confirmar Eliminación</h5>
        </div>
        <div class="modal-body">
            ¿Estás seguro de eliminar este código?
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="cerrarModal('confirmEliminarModal')">Cancelar</button>
            <a href="#" class="btn btn-danger" id="confirmEliminarBtn">Eliminar</a>
        </div>
    </div>
</div>

<script>
// Copiar código al portapapeles
function copiarCodigo(codigo, boton) {
    navigator.clipboard.writeText(codigo).then(() => {
        boton.innerText = 'Copiado ✔';
        setTimeout(() => { boton.innerText = 'Copiar'; }, 2000);
    });
}

// Abrir modal de confirmación para eliminar
function abrirModalEliminar(id) {
    var modal = document.getElementById('confirmEliminarModal');
    var btn = document.getElementById('confirmEliminarBtn');
    btn.href = 'gestionar_codes.php?eliminar=' + id;
    modal.style.display = 'block';
}

// Cerrar modal
function cerrarModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}
</script>

<?php include('includes/footer.php'); ?>